from flask import Flask, request
import requests
import re
import json
from config import BOT_TOKEN, GROUP_CHAT_ID, CHANNEL_CHAT_ID, API_URL, API_SECRET_TOKEN

app = Flask(__name__)
TELEGRAM_API = f"https://api.telegram.org/bot{BOT_TOKEN}"

user_states = {}

def send_message(chat_id, text, reply_markup=None):
    data = {
        "chat_id": chat_id,
        "text": text,
        "parse_mode": "Markdown"
    }
    if reply_markup:
        data["reply_markup"] = json.dumps(reply_markup)
    requests.post(f"{TELEGRAM_API}/sendMessage", data=data)

def validate_email(email):
    pattern = r"^[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}$"
    return re.match(pattern, email)

def check_membership(user_id):
    try:
        group = requests.get(f"{TELEGRAM_API}/getChatMember", params={
            "chat_id": GROUP_CHAT_ID,
            "user_id": user_id
        }).json()

        channel = requests.get(f"{TELEGRAM_API}/getChatMember", params={
            "chat_id": CHANNEL_CHAT_ID,
            "user_id": user_id
        }).json()

        group_ok = group.get("result", {}).get("status") in ["member", "administrator", "creator"]
        channel_ok = channel.get("result", {}).get("status") in ["member", "administrator", "creator"]

        return group_ok and channel_ok
    except Exception as e:
        print(f"Error checking membership: {e}")
        return False

@app.route(f"/{BOT_TOKEN}", methods=["POST"])
def webhook():
    update = request.get_json()

    if "message" in update:
        chat_id = update["message"]["chat"]["id"]
        user_id = update["message"]["from"]["id"]
        text = update["message"].get("text", "").strip()

        if text == "/start":
            send_message(chat_id, "👋 Welcome! Use /verify to verify your email.")
        
        elif text == "/verify":
            if not check_membership(user_id):
                join_buttons = {
                    "inline_keyboard": [
                        [{"text": "Join SVP Group", "url": "https://t.me/+KXHIn8vU5G4zMTNh"}],
                        [{"text": "Join SVP Channel", "url": "https://t.me/+0WQkWfyFmGczMjdh"}],
                        [{"text": "✅ I've Joined", "callback_data": "joined"}]
                    ]
                }
                send_message(chat_id, "❌ Please join both the group and channel to continue.", join_buttons)
                return "ok", 200

            user_states[user_id] = "awaiting_email"
            send_message(chat_id, "✅ Please enter your *registered email address* to verify.")
        
        elif user_states.get(user_id) == "awaiting_email":
            if not validate_email(text):
                send_message(chat_id, "❌ That doesn't look like a valid email. Please try again.")
                return "ok", 200

            payload = {
                "telegram_id": user_id,
                "username": update["message"]["from"].get("username", "unknown"),
                "email": text,
                "token": API_SECRET_TOKEN
            }

            headers = {"Content-Type": "application/json"}
            try:
                resp = requests.post(API_URL, json=payload, headers=headers)
                data = resp.json()
            except Exception as e:
                send_message(chat_id, "⚠️ Server error. Try again later.")
                return "ok", 200

            msg = {
                "reward_granted": "🎉 Thank you! Your reward has been granted.",
                "already_rewarded": "✅ You have already claimed your reward.",
                "not_registered": "❌ Your email is not registered. Please sign up first.",
            }.get(data.get("status"), f"⚠️ {data.get('message', 'Unknown error')}")

            send_message(chat_id, msg)
            user_states[user_id] = None

    elif "callback_query" in update:
        query = update["callback_query"]
        user_id = query["from"]["id"]
        chat_id = query["message"]["chat"]["id"]

        if query["data"] == "joined":
            if check_membership(user_id):
                user_states[user_id] = "awaiting_email"
                send_message(chat_id, "🎉 You've joined! Now send your *registered email address* to continue.")
            else:
                send_message(chat_id, "❌ Still not in both group and channel. Please join both.")

    return "ok", 200

@app.route("/")
def home():
    return "Bot is running!", 200
